/*
 * MN2WS SPI controller driver.
 *
 * Maintainer: 
 *
 * Copyright (C) 2010 
 *
 * CPM SPI and QE buffer descriptors mode support:
 * Copyright (c) 2009  MontaVista Software, Inc.
 * Author: Anton Vorontsov <avorontsov@ru.mvista.com>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 */


#include <linux/module.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/spi/spi.h>
#include <linux/spi/spidev.h>
#include <linux/spi/spi-mn2ws-internal.h>
#include <linux/platform_device.h>
#include <asm/unaligned.h>
#include <linux/spi/spi-mn2ws.h>
#include <unit/spi-mn2ws-unit.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <mach/intctl-regs.h>

#define SPI_N_CHIPSEL   2


/* Driver internal data */
struct mn2ws_spi_info{
	u16 num_chipselect;
	u32	tclk;		/* no <linux/clk.h> support yet */
	u32	enable_clock_fix;
};

static struct spi_mn2ws_pdata spi_mn2ws_pdata_0 = {
	.base = SPI_MN2WS_CH0_BASE,
	.irq  = SPI_MN2WS_CH0_IRQ,
};

static struct spi_mn2ws_pdata spi_mn2ws_pdata_1 = {
	.base = SPI_MN2WS_CH1_BASE,
	.irq  = SPI_MN2WS_CH1_IRQ,
};

static struct spi_board_info mn2ws_spi_boardinfo_0 = {
	.bus_num = 0,
	.chip_select = 0,
	.max_speed_hz = 50000000,
	.modalias = "spidev",
	.platform_data = &spi_mn2ws_pdata_0,
};

static struct spi_board_info mn2ws_spi_boardinfo_1 = {
	.bus_num = 1,
	.chip_select = 1,
	.max_speed_hz = 50000000,
	.modalias = "spidev",
	.platform_data = &spi_mn2ws_pdata_1,
};

struct mn2ws_spi{
	struct work_struct	work;
	struct spi_master	*master;
	spinlock_t		lock;
	
	struct list_head	msg_queue;
	struct mn2ws_spi_info	*spi_info;
	void __iomem		*base;
};

struct spi_mn2ws_data *spi_g_mn2ws_data[SPI_MN2WS_ADAP_MAX ];
struct spi_ioctl_cs *spi_g_mn2ws_cs[SPI_MN2WS_ADAP_MAX ] = {NULL,NULL};

static struct workqueue_struct *mn2ws_spi_wq;

int spi_g_word_len[SPI_MN2WS_ADAP_MAX ];

/*
static inline void __iomem *spi_reg(struct mn2ws_spi *mn2ws_spi, u32 reg)
{
	return mn2ws_spi->base + reg;
}
*/
static void
spi_mn2ws_write32(
	struct spi_mn2ws_data *spi_data, unsigned long reg, unsigned long val
)
{
	void *addr;

	if (!spi_data){
		return;
	}
	addr = spi_data->iomap_base + reg ;
	writel(val, addr);	/* iowrite32(val, addr); */
}
EXPORT_SYMBOL_GPL(spi_mn2ws_write32);

static unsigned long
spi_mn2ws_read32(struct spi_mn2ws_data *spi_data, unsigned long reg)
{
	void *addr;
	unsigned long val ;

	if (!spi_data){
		return 0;
	}
	addr = spi_data->iomap_base + reg ;
	val  = readl(addr);	/* ioread32(addr); */

	return val;
}
EXPORT_SYMBOL_GPL(spi_mn2ws_read32);

#ifdef REGCHK
static unsigned long
spi_mn2ws_read_verify(struct spi_mn2ws_data *spi_data, unsigned long reg)
{
	void *addr;
	unsigned long val ;

	if (!spi_data){
		return 0;
	}
	addr = spi_data->iomap_base + reg ;
	do {
		val  = readl(addr);	/* ioread32(addr); */
	} while (val == 0);

	return val;
}
EXPORT_SYMBOL_GPL(spi_mn2ws_read_verify);

static void
spi_mn2ws_write_verify(
	struct spi_mn2ws_data *spi_data, unsigned long reg, unsigned long val
)
{
	void *addr;
	unsigned long read_val;

	if (!spi_data){
		return;
	}
	if (val == 0){
		printk ("\n####    WRITE DATA = 0 ###\n");
		return;
	}
	addr = spi_data->iomap_base + reg ;
	do {
		writel(val, addr);
		read_val = spi_mn2ws_read_verify(spi_data, reg);
	}while (read_val != val);
}
EXPORT_SYMBOL_GPL(spi_mn2ws_write_verify);
#endif

static u16 mn2ws_spi_set_mask(int item, int txrx){
	u16 mask = 0;
	if(SPI_MODE == item){
		mask = SPI_MN2WS_CKS_CKPHS | SPI_MN2WS_CKS_CKINIT;
	}else if(LSB_FIRST == item){
		mask = 0x0040;
	}else if(BITS_PER_WORD == item){
		if(MN2WS_RX == txrx){
			mask = SPI_MN2WS_RXWDS_DTLEN;
		}else if(MN2WS_TX == txrx){
			mask = SPI_MN2WS_TXWDS_DTLEN;
		}
	}else if(DATA_ORDER == item){
		mask = 0x0080;
	}else if(DATA_DELAY == item){
		if(MN2WS_RX == txrx){
			mask = SPI_MN2WS_RXWDS_DTDLY;
		}else if(MN2WS_TX == txrx){
			mask = SPI_MN2WS_TXWDS_DTDLY;
		}
	}else if(WD_LEN == item){
		mask = SPI_MN2WS_TXWDS_WDLEN;
	}else if(FIFO_TH == item){
		if(MN2WS_RX == txrx){
			mask = SPI_MN2WS_FC_RXFTH;
		}else if(MN2WS_TX == txrx){
			mask = SPI_MN2WS_FC_TXFTH;
		}
	}else if(CLK_DELAY == item){
		mask = SPI_MN2WS_CKS_CKDLY;
	}else if(CLK_RATE == item){
		mask = SPI_MN2WS_CKS_CKRAT;
	}else if(CS_POL == item){
		mask = SPI_MN2WS_FPS_FSPOL;
	}else if(CS_STRT == item){
		mask = SPI_MN2WS_FPS_FSTRT;
	}else if(P_WDTH == item){
		mask = SPI_MN2WS_FPS_PWDTH;
	}else if(DATA_FORMAT == item){
		if(MN2WS_RX == txrx){
			mask = SPI_MN2WS_RXWDS_RDTF;
		}else if(MN2WS_TX == txrx){
			mask = SPI_MN2WS_TXWDS_TDTF;
		}
	}
	
	return mask;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_set_mask);

static int mn2ws_spi_set_offset(int item, int txrx){
	int offset = 0;
	if(MN2WS_RX == txrx){
		if(DATA_DELAY == item || LSB_FIRST == item || DATA_ORDER == item || BITS_PER_WORD == item || DATA_FORMAT == item){
			offset = SPI_MN2WS_RXWDS;
		}else if(item == FIFO_TH){
			offset = SPI_MN2WS_FC;
		}else{
			return -EINVAL;
		}
	}else if(MN2WS_TX == txrx){
		if(DATA_DELAY == item || LSB_FIRST == item || DATA_ORDER == item || BITS_PER_WORD == item || WD_LEN == item || DATA_FORMAT == item){
			offset = SPI_MN2WS_TXWDS;
		}else if(FIFO_TH == item){
			offset = SPI_MN2WS_FC;
		}else{
			return -EINVAL;
		}
	}else if(MN2WS_TXRX == txrx){
		if(CLK_DELAY == item || CLK_RATE == item){
			offset = SPI_MN2WS_CKS;
		}else if(CS_POL == item || CS_STRT == item){
			offset = SPI_MN2WS_FPS;
		}else if(SPI_MODE == item){
			offset = SPI_MN2WS_CKS;
		}else if(P_WDTH == item){
			offset = SPI_MN2WS_FPS;
		}else{
			return -EINVAL;
		}
	}
	
	return offset;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_set_offset);

/* mn2ws_spi_set_reg쥸ͤꤹ						 */
/* @spi_data	:													 */
/* @item		:ꤷ쥸									 */
/* @txrx		:TX(MN2WS_TX)RX(MN2WS_RX)ξ(MN2WS_TXRX)		 */
/* @set_val		:												 */
int mn2ws_spi_set_reg(struct spi_device *spi, int item, int txrx, int set_val){
	int status = SPI_STATUS_OK;
	u16 mask;
	u16 mask2;
	u16 setting = 0;
	u16 setting2 = 0;
	u16 length = 0;
	u16 i = spi->chip_select;

	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[i];
	
	int offset = 0;
	
	/* set address */
	offset = mn2ws_spi_set_offset(item, txrx);
	
	/* set mask */
	mask = mn2ws_spi_set_mask(item, txrx);
	
	/* read register */
#ifdef REGCHK
	setting = spi_mn2ws_read_verify(spi_data, offset);
    printk(KERN_INFO"reg check \n");
#else
	setting = spi_mn2ws_read32(spi_data, offset);
#endif
	/* set value */
	/* for MODE */
	if(SPI_MODE == item){
		if(MN2WS_MODE3 == set_val){
			setting |= mask;						//11
		}else{
			setting &= ~mask;						//00
			if(MN2WS_MODE1 == set_val){
				setting += SPI_MN2WS_CKS_CKPHS;		//10
			}else if(MN2WS_MODE2 == set_val){
				setting += SPI_MN2WS_CKS_CKINIT;	//01
			}
		}
	/* for LSB_FIRST */
	}else if(LSB_FIRST == item){
		if(MN2WS_MSB == set_val){
			setting &= ~mask;
		}else if(MN2WS_LSB == set_val){
			setting |= mask;
		}
	}else if(BITS_PER_WORD == item){
		/* word length */
		length = setting;
		length &= 0x3f00;
		/* data length */
		setting &= ~mask;
		setting += set_val;
		
	/* for DATA_ORDER */
	}else if(DATA_ORDER == item){
		if(MN2WS_F_STUFF == set_val){
			setting &= ~mask;
		}else if(MN2WS_P_STUFF == set_val){
			setting |= mask;
		}
	/* for DATA_DELAY */
	}else if(DATA_DELAY == item){
		if(0 == set_val){			//00
			setting &= ~mask;
		}else if(3 <= set_val){		//11
			return -EINVAL;
		}else{						//01,10
			setting &= ~mask;
			setting += (set_val << 14);
		}
	/* for WD_LEN */
	}else if(WD_LEN == item){
		/* data length */
		spi_g_word_len[i] = set_val;
		length = setting;
		length &= 0x3f;
		/* word length */
		mask2 = SPI_MN2WS_FPS_PWDTH;
#ifdef REGCHK
		setting2 = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_FPS);
#else
		setting2 = spi_mn2ws_read32(spi_data, SPI_MN2WS_FPS);
#endif
		setting2 &= ~mask2;
		setting2 += set_val;
#ifdef REGCHK
		spi_mn2ws_write_verify(spi_data, SPI_MN2WS_FPS, setting2);
#else
		spi_mn2ws_write32(spi_data, SPI_MN2WS_FPS, setting2);
#endif
		setting &= ~mask;
		setting += (set_val << 8);
		
	/* for CLK_DELAY */
	}else if(CLK_DELAY == item){
		if(MN2WS_C_DLY == set_val){
			setting &= ~mask;
		}else if(MN2WS_C_NODLY == set_val){
			setting |= mask;
		}
	/* for CLK_RATE */
	}else if(CLK_RATE == item){
		setting &= ~mask;
		setting += set_val;
	/* for CS_POL */
	}else if(CS_POL == item){
		if(MN2WS_CSPOL_PEDGE == set_val){
			setting &= ~mask;
		}else if(MN2WS_CSPOL_NEDGE == set_val){
			setting |= mask;
		}
	/* for CS_STRT */
	}else if(CS_STRT == item){
		if(MN2WS_CSSTRT_PEDGE == set_val){
			setting &= ~mask;
		}else if(MN2WS_CSSTRT_NEDGE == set_val){
			setting |= mask;
		}
	/* for FIFO_TH */
	}else if(FIFO_TH == item){
		if(MN2WS_TX == txrx){
			setting &= ~mask;
			setting += (set_val << 8);
		}else{
			setting &= ~mask;
			setting += set_val;
		}
	}else if(P_WDTH == item){
		/* pulse width */
		setting &= ~mask;
		setting |= set_val;
	}else if(DATA_FORMAT == item){
		/* pulse width */
		setting &= ~mask;
		setting |= (set_val<<6);
	}
	/* write register */
//	writel(setting, addr);
#ifdef REGCHK
	spi_mn2ws_write_verify(spi_data, offset, setting);
#else
	spi_mn2ws_write32(spi_data, offset, setting);
#endif
	return status;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_set_reg);

/* mn2ws_spi_get_reg쥸ͤ						 */
/* @spi_data	:													 */
/* @item		:ɤ߽Ф쥸								 */
/* @txrx		:TX(MN2WS_TX)RX(MN2WS_RX)ξ(MN2WS_TXRX)		 */
/* @get_val		:ǼѥХåե										 */
int mn2ws_spi_get_reg(struct spi_device *spi, int item, int txrx, int *get_val){
	int status = SPI_STATUS_OK;
	u16 mask;
	u16 val = 0;
	u16 i = spi->chip_select;

	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[i];

	int offset = 0;

	/* set address */
	offset = mn2ws_spi_set_offset(item, txrx);
	
	/* set mask */
	mask = mn2ws_spi_set_mask(item, txrx);
	
	/* read register */
#ifdef REGCHK
	val = spi_mn2ws_read_verify(spi_data, offset);
#else
	val = spi_mn2ws_read32(spi_data, offset);
#endif
	val &= mask;
	
	/* SPI Mode */
	if(SPI_MODE == item){
		val = val >> 13;
		if(2 == val){
			val = 1;
		}else if(1 == val){
			val = 2;
		}
	/* LSB First */
	}else if(LSB_FIRST == item){
		val = val >> 6;
	/* Clock Delay */
	}else if(CLK_DELAY == item){
		val = val >> 12;
	/* Clock Rate */
	}else if(CLK_RATE == item){
	/* CS Pol */
	}else if(CS_POL == item){
		val = val >> 15;
	/* CS Start */
	}else if(CS_STRT == item){
		val = val >> 14;
	/* Word Length */
	}else if(WD_LEN == item){
		val = val >> 8;
	/* Data Order */
	}else if(DATA_ORDER == item){
		val = val >> 7;
	}else if(DATA_DELAY == item){
		val = val >> 14;
	/* FIFO Threshold */
	}else if(FIFO_TH == item){
		if(MN2WS_TX == txrx){
			val = val >> 8;
		}
	}
	
	*get_val = val;
	
	return status;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_get_reg);

extern
int mn2ws_spi_device_check(struct spi_device *spi )
{
	int status = SPI_STATUS_OK;
	u16 i = spi->chip_select;

	if (spi_g_mn2ws_data[i] == NULL){
		status = -ENODEV;
	}
 	return status;	
}
EXPORT_SYMBOL_GPL(mn2ws_spi_device_check);

extern
void mn2ws_spi_init_register(struct spi_device *spi )
{
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];
    printk(KERN_INFO"mn2ws init register spi  cs %d \n",spi->chip_select);
#ifdef REGCHK
	spi_mn2ws_write_verify(spi_data, SPI_MN2WS_CKS, SPI_MN2WS_CKS_INIT);
    printk(KERN_INFO"mn2ws init register spi  cks  \n");
	spi_mn2ws_write_verify(spi_data, SPI_MN2WS_TXWDS, SPI_MN2WS_TXWDS_INIT);
	printk(KERN_INFO"mn2ws init register spi  txwds \n");
	spi_mn2ws_write_verify(spi_data, SPI_MN2WS_RXWDS, SPI_MN2WS_RXWDS_INIT);
	printk(KERN_INFO"mn2ws init register spi  rxwds \n");
	spi_mn2ws_write_verify(spi_data, SPI_MN2WS_FPS, SPI_MN2WS_FPS_INIT);
	printk(KERN_INFO"mn2ws init register spi  fps  \n");
	spi_mn2ws_write_verify(spi_data, SPI_MN2WS_FC, SPI_MN2WS_FC_INIT);
	printk(KERN_INFO"mn2ws init register spi  fc  \n");
#else
	spi_mn2ws_write32(spi_data, SPI_MN2WS_CKS, SPI_MN2WS_CKS_INIT);
	printk(KERN_INFO"mn2ws init register spi  cks \n");

	spi_mn2ws_write32(spi_data, SPI_MN2WS_TXWDS, SPI_MN2WS_TXWDS_INIT);
	printk(KERN_INFO"mn2ws init register spi  txwds \n");
	spi_mn2ws_write32(spi_data, SPI_MN2WS_RXWDS, SPI_MN2WS_RXWDS_INIT);
	printk(KERN_INFO"mn2ws init register spi  rxwds \n");
	spi_mn2ws_write32(spi_data, SPI_MN2WS_FPS, SPI_MN2WS_FPS_INIT);
	printk(KERN_INFO"mn2ws init register spi  fps \n");
	spi_mn2ws_write32(spi_data, SPI_MN2WS_FC, SPI_MN2WS_FC_INIT);
	printk(KERN_INFO"mn2ws init register spi  fc \n");
#endif
}
EXPORT_SYMBOL_GPL(mn2ws_spi_init_register);

/* Chip Select */
#ifdef PORT_CS
void spi_cs_assert(struct spi_device *spi )
{
	void *addr;
	unsigned long val;
	unsigned long val_org;
	unsigned long cspol;
	u16 i = spi->chip_select;

	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[i];

#ifdef REGCHK
	cspol = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_FPS);
#else
	cspol = spi_mn2ws_read32(spi_data, SPI_MN2WS_FPS);
#endif
	cspol &= SPI_MN2WS_FPS_FSPOL;
	addr = spi_data->p_iomap_base + spi_data->port_iod  ;
	val_org = readb(addr);
	if (cspol == SPI_MN2WS_FPS_FSPOL){
		val  = spi_data->cs_high & val_org;
	} else {
		val  = spi_data->cs_low | val_org;
	}
	writeb(val,addr);
}
EXPORT_SYMBOL_GPL(spi_cs_assert);

void spi_cs_negate(struct spi_device *spi )
{
	void *addr;
	unsigned long val;
	unsigned long val_org;
	unsigned long cspol;
	u16 i = spi->chip_select;

	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[i];

#ifdef REGCHK
	cspol = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_FPS);
#else
	cspol = spi_mn2ws_read32(spi_data, SPI_MN2WS_FPS);
#endif
	cspol &= SPI_MN2WS_FPS_FSPOL;
	addr = spi_data->p_iomap_base + spi_data->port_iod  ;
	printk(KERN_INFO"readb cs negate \n");
	val_org = readb(addr);
	if (cspol == SPI_MN2WS_FPS_FSPOL){
		printk(KERN_INFO"cspol is equal \n");
		val  = spi_data->cs_low | val_org;
	} else {
	    printk(KERN_INFO"cspol is not equal \n");
		val  = spi_data->cs_high & val_org;
	}
	printk(KERN_INFO"writeb cs negate \n");
	writeb(val,addr);
}
EXPORT_SYMBOL_GPL(spi_cs_negate);

void spi_cs_set(struct spi_device *spi )
{
	u16 i = spi->chip_select;

	if (spi_g_mn2ws_cs[i] == NULL){
		spi_g_mn2ws_cs[i] = kzalloc(sizeof(*spi_g_mn2ws_cs[i]), GFP_KERNEL);
	}
	spi_g_mn2ws_cs[i]->cs = spi->cs;
	spi_g_mn2ws_cs[i]->next = spi->next;

	if ((spi_g_mn2ws_cs[i]->cs == CS_ON) && (spi_g_mn2ws_cs[i]->next == NEXT_OFF)) {
		spi_cs_assert(spi);
	}
	else if ((spi_g_mn2ws_cs[i]->cs == CS_OFF) && (spi_g_mn2ws_cs[i]->next == NEXT_OFF)) {
		spi_cs_negate(spi);
	}

}
EXPORT_SYMBOL_GPL(spi_cs_set);
#endif

int mn2ws_spi_get_clkrate(struct spi_device *spi ){
	unsigned int clkrate;
	int rate;

	mn2ws_spi_get_reg(spi, CLK_RATE, MN2WS_TXRX, &rate);

	clkrate = ( SPI_CLK / (rate * 10 * 1000));

	return clkrate;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_get_clkrate);

static irqreturn_t
spi_mn2ws_handler(int this_irq, void *dev_id )
{
	unsigned long flags;
	struct spi_device  *spi=dev_id;

	flags = 0;
	spin_lock_irqsave(&(spi_g_mn2ws_data[spi->chip_select]->lock), flags);
	spi_g_mn2ws_data[spi->chip_select]->irq_event = 1;
	spin_unlock_irqrestore(&(spi_g_mn2ws_data[spi->chip_select]->lock), flags);
	wake_up(&(spi_g_mn2ws_data[spi->chip_select]->wait));

	return IRQ_HANDLED;
}

static int mn2ws_spi_setup(struct spi_device *spidev)
{
	struct mn2ws_spi *mn2ws_spi;
	
	mn2ws_spi = spi_master_get_devdata(spidev->master);
	
	return 0;
}

static void 
fifo_clear(struct spi_device *spi)
{
	unsigned long val  = 0;
	int status = SPI_STATUS_OK;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];

#ifdef REGCHK
	val = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_FC);
#else
	val = spi_mn2ws_read32(spi_data, SPI_MN2WS_FC);
#endif
	do {
		spi_mn2ws_write32(spi_data, SPI_MN2WS_FC, val | SPI_MN2WS_FC_RXFFL);
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
	}while (status != SPI_STATUS_OK);
}

static inline int
mn2ws_spi_write_read_8bit(struct spi_device *spi, const u8 **tx_buf, u8 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];
	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC  );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if ((tx_buf && *tx_buf) &&  (spi->xfer == SPI_TX)){
		val = *(*tx_buf)++ ;	
#ifdef REGCHK
		do{
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif

	count = 0;
	status = SPI_STATUS_OK;

		do {
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
			status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
			status &= SPI_MN2WS_SR_RNE;
			count ++;
			if (count > SPI_RX_COUNT_MAX){
				printk ("\n####    TRANSFER RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
			}
		}while (status == SPI_STATUS_OK);

	}else if (spi->xfer == SPI_RX) {
		fifo_clear(spi);
#ifdef REGCHK
		do{
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, TX_DUMMY_DATA);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	if (spi->xfer == SPI_TX){
		return 1;
	}

	count = 0;
	status = SPI_STATUS_OK;

	do{
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);

#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RECEIVE RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	count = 0;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do{
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		*(*rx_buf)++ = val; 
	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_8bit);

static inline int
mn2ws_spi_write_read_16bit(struct spi_device *spi, const u16 **tx_buf, u16 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];
#if 0	
	int tx_count =0;
#endif
	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if ((tx_buf && *tx_buf) &&  (spi->xfer == SPI_TX)){
		val = get_unaligned((*tx_buf)++);
#ifdef REGCHK
		do {
#endif		
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
#if 0
		tx_count++;
		printk ("####tx_count = %x\n",tx_count);;
#endif
		count = 0;
		status = SPI_STATUS_OK;

		do {
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
			status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
			status &= SPI_MN2WS_SR_RNE;
			count ++;
			if (count > SPI_RX_COUNT_MAX){
				printk ("\n####    TRANSFER RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
			}
		}while (status == SPI_STATUS_OK);

	}else if (spi->xfer == SPI_RX) {
		fifo_clear(spi);
#ifdef REGCHK
		do {
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, TX_DUMMY_DATA);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	if (spi->xfer == SPI_TX){
		return 1;
	}

	count = 0;
	status = SPI_STATUS_OK;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RECEIVE RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	count = 0;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do {
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		put_unaligned(val , (*rx_buf)++);
	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_16bit);

static inline int
mn2ws_spi_write_read_32bit(struct spi_device *spi, const u32 **tx_buf, u32 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];

	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if ((tx_buf && *tx_buf) &&  (spi->xfer == SPI_TX)){
		val = get_unaligned((*tx_buf)++);
#ifdef REGCHK
		do {
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif

		count = 0;
		status = SPI_STATUS_OK;

		do {
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
			status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
			status &= SPI_MN2WS_SR_RNE;
			count ++;
			if (count > SPI_RX_COUNT_MAX){
				printk ("\n####    TRANSFER RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
			}
		}while (status == SPI_STATUS_OK);

	}else if (spi->xfer == SPI_RX) {
		fifo_clear(spi);
#ifdef REGCHK
		do {
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, TX_DUMMY_DATA);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	if (spi->xfer == SPI_TX){
		return 1;
	}

	count = 0;
	status = SPI_STATUS_OK;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RECEIVE RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	count = 0;
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do {
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		put_unaligned(val , (*rx_buf)++);
	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_32bit);

static inline int
mn2ws_spi_write_read_8bit_full_duplex(struct spi_device *spi, const u8 **tx_buf, u8 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];

	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do{
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	fifo_clear(spi);
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (tx_buf && *tx_buf){
		val = *(*tx_buf)++ ;	
#ifdef REGCHK
		do{
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	count = 0;
	status = SPI_STATUS_OK;

	do{
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);

	status = SPI_STATUS_OK;
	count = 0;

	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do {
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		*(*rx_buf)++ = val;

	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_8bit_full_duplex);

static inline int
mn2ws_spi_write_read_16bit_full_duplex(struct spi_device *spi, const u16 **tx_buf, u16 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];

	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	fifo_clear(spi);
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (tx_buf && *tx_buf){
		val = get_unaligned((*tx_buf)++);
#ifdef REGCHK
		do {
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	count = 0;
	status = SPI_STATUS_OK;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);

	status = SPI_STATUS_OK;
	count = 0;

	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do {
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		put_unaligned(val , (*rx_buf)++);
	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_16bit_full_duplex);

static inline int
mn2ws_spi_write_read_32bit_full_duplex(struct spi_device *spi, const u32 **tx_buf, u32 **rx_buf )
{
	unsigned int val ;
	int status;
	int reg_status;
	int count;
	struct spi_mn2ws_data *spi_data = spi_g_mn2ws_data[spi->chip_select];

	val = 0;
	status = SPI_STATUS_OK;
	reg_status = SPI_STATUS_OK;
	count = 0;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_TNF;
		count ++;
		if (count > SPI_TX_COUNT_MAX){
			printk ("\n####    TX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);
	status = SPI_STATUS_OK;
	fifo_clear(spi);
#ifdef REGCHK
	do{
#endif
		spi_mn2ws_write32(spi_data, SPI_MN2WS_IC, SPI_MN2WS_IC_TCIC );
#ifdef REGCHK
		reg_status = spi_mn2ws_read32(spi_data, SPI_MN2WS_IS) & 0x001f;
	}while (reg_status != SPI_STATUS_OK);
#endif
	if (tx_buf && *tx_buf){
		val = get_unaligned((*tx_buf)++);
#ifdef REGCHK
		do {
#endif
			spi_mn2ws_write32(spi_data, SPI_MN2WS_TXDR, val);
#ifdef REGCHK
			while (1){
				if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_TFE){
					break;
				}
			}
			if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR) & SPI_MN2WS_SR_BUSY){
				while(1){
					if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
						break;
					}
				}
				break;
			}else if (spi_mn2ws_read_verify(spi_data, SPI_MN2WS_IS) & SPI_MN2WS_IS_TCID){
				break;
			}
		} while ( 1 );
#endif
	}

	count = 0;
	status = SPI_STATUS_OK;

	do {
#ifdef REGCHK
		status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
#else 
		status = spi_mn2ws_read32(spi_data, SPI_MN2WS_SR);
#endif
		status &= SPI_MN2WS_SR_RNE;
		count ++;
		if (count > SPI_RX_COUNT_MAX){
			printk ("\n####    RX BUFFER CHECK TIMEOUT    ###\n");
			return -ETIMEDOUT;
		}
	}while (status == SPI_STATUS_OK);

	status = SPI_STATUS_OK;
	count = 0;

	if (rx_buf && *rx_buf){
#ifdef REGCHK
		do {
#endif
			val = spi_mn2ws_read32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_RXDR );
#ifdef REGCHK
			status = spi_mn2ws_read_verify(spi_data, SPI_MN2WS_SR);
			status &= SPI_MN2WS_SR_RNE;
		}while (status != SPI_STATUS_OK);
#endif
		put_unaligned(val , (*rx_buf)++);
	}
	return 1;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read_32bit_full_duplex);

static unsigned int
mn2ws_spi_write_read(struct spi_device *spi, struct spi_transfer *xfer)
{
	int count;
	unsigned int sub_num;
	int word_len;
	int status = SPI_STATUS_OK;
	    
	count = xfer->len;
	printk(KERN_INFO"xfer length %d \n",count);
	word_len = spi_g_word_len[spi->chip_select];
	spi_mn2ws_write32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_CTL, SPI_MN2WS_CTL_SSIEN);

	if (word_len <= 8) {
		const u8 *tx = xfer->tx_buf;
		u8 *rx = xfer->rx_buf;
		do {
#ifdef REGCHK
			spi_mn2ws_write_verify(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#else
			spi_mn2ws_write32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#endif
			if (spi->xfer == SPI_FULL_DUPLEX){
				status = mn2ws_spi_write_read_8bit_full_duplex(spi, &tx, &rx);
			}else{
			status = mn2ws_spi_write_read_8bit(spi, &tx, &rx);
			}
			if (status < SPI_STATUS_OK){
				return status;
			}
			count--;
		}
		
		while (count);
	} else if (word_len > 8 && word_len <= 16) {
		const u16 *tx = xfer->tx_buf;
		u16 *rx = xfer->rx_buf;
		sub_num = 2;

		do{
#ifdef REGCHK
			spi_mn2ws_write_verify(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#else
			spi_mn2ws_write32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#endif
			if (spi->xfer == SPI_FULL_DUPLEX){
				status = mn2ws_spi_write_read_16bit_full_duplex(spi, &tx, &rx);
			}else{
				status = mn2ws_spi_write_read_16bit(spi, &tx, &rx);
			}
			if (status < SPI_STATUS_OK){
				return status;
			}
			if (count < sub_num){
				count -= 1;
			}else{
				count -= sub_num;
			}
		}while (count);
	} else {
		const u32 *tx = xfer->tx_buf;
		u32 *rx = xfer->rx_buf;
		if (word_len > 16 && word_len <= 24) {
			sub_num = 3;
		} else if (word_len > 24 && word_len <= 32) {
			sub_num = 4;
		} else if (word_len > 32 && word_len <= 40) {
			sub_num = 5;
		} else if (word_len > 40 && word_len <= 48) {
			sub_num = 6;
		} else if (word_len > 48 && word_len <= 54) {
			sub_num = 7;
		} else {
			sub_num = 8;
		}

		do{
#ifdef REGCHK
			spi_mn2ws_write_verify(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#else
			spi_mn2ws_write32(spi_g_mn2ws_data[spi->chip_select], SPI_MN2WS_IE, SPI_MN2WS_IE_TCIE);
#endif
			if (spi->xfer == SPI_FULL_DUPLEX){
			status = mn2ws_spi_write_read_32bit_full_duplex(spi, &tx, &rx);
			} else {
				status = mn2ws_spi_write_read_32bit(spi, &tx, &rx);
			}
			if (status < SPI_STATUS_OK){
				return status;
			}
			if (count < sub_num){
				count -= count;
			} else {
				count -= sub_num;
			}
		} while (count);
	}
	return xfer->len - count;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_write_read);

static void mn2ws_spi_work(struct work_struct *work){
	struct mn2ws_spi *mn2ws_spi;
	struct spi_device *spi;
	u8 cs;
	u8 next;
	unsigned long flags;
    printk(KERN_INFO"SPI WORK \n");
	flags = 0;

	mn2ws_spi = container_of(work, struct mn2ws_spi, work);

	spin_lock_irqsave((&mn2ws_spi->lock), flags);
	while (!list_empty(&mn2ws_spi->msg_queue)) {
		struct spi_message *message;
		struct spi_transfer *spi_transfer = NULL;

		message = container_of(mn2ws_spi->msg_queue.next, struct spi_message, queue);

		list_del_init(&message->queue);
		spin_unlock_irqrestore((&mn2ws_spi->lock), flags);

		spi = message->spi;

#ifdef PORT_CS
		/* chip select assert */
		if (spi_g_mn2ws_cs[spi->chip_select] == NULL){
			cs = CS_OFF;
			next = NEXT_OFF;
		}else{
			cs = spi_g_mn2ws_cs[spi->chip_select]->cs;
			next = spi_g_mn2ws_cs[spi->chip_select]->next;
		}
#endif

		/* Load defaults */

		list_for_each_entry(spi_transfer, &message->transfers, transfer_list) {
#ifdef PORT_CS
			spi_cs_assert(spi);
			if( next ){
				spi->next = NEXT_OFF ;
			}		
#endif
			if (spi_transfer->len){
				printk(KERN_INFO"SPI WORK write read\n");
				message->actual_length += mn2ws_spi_write_read(spi, spi_transfer);
			}
		}

		message->complete(message->context);

		spin_lock_irqsave((&mn2ws_spi->lock), flags);
	}

	spin_unlock_irqrestore((&mn2ws_spi->lock), flags);
}
EXPORT_SYMBOL_GPL(mn2ws_spi_work);

int mn2ws_spi_transfer(struct spi_device *spidev, struct spi_message *msg)
{
	struct mn2ws_spi *mn2ws_spi;
	struct spi_transfer *spi_transfer = NULL;
	unsigned long flags;

	msg->actual_length = 0;
	msg->status = SPI_STATUS_OK;
	flags = 0;
    printk(KERN_INFO"spi tranfer \n");
	/* reject invalid messages and transfers */
	if (list_empty(&msg->transfers) || !msg->complete){
		return -EINVAL;
	}

	mn2ws_spi = spi_master_get_devdata(spidev->master);

	list_for_each_entry(spi_transfer, &msg->transfers, transfer_list) {

		if (spi_transfer->tx_buf == NULL && spi_transfer->rx_buf == NULL && spi_transfer->len) {
			dev_err(&spidev->dev,
			"message rejected : "
			"invalid transfer data buffers\n");
			goto msg_rejected;
		}
	}

	spin_lock_irqsave(&mn2ws_spi->lock, flags);
	list_add_tail(&msg->queue, &mn2ws_spi->msg_queue);
	queue_work(mn2ws_spi_wq, &mn2ws_spi->work);
	queue_work(mn2ws_spi_wq, &mn2ws_spi->work);
	spin_unlock_irqrestore(&mn2ws_spi->lock, flags);
	
	return 0;
msg_rejected:
	/* Message rejected and not queued */
	msg->status = -EINVAL;
	if (msg->complete){
		msg->complete(msg->context);
	}
	return -EINVAL;
}
EXPORT_SYMBOL_GPL(mn2ws_spi_transfer);

/******************************************************************************
 * Data
 *****************************************************************************/
#ifdef CONFIG_SPI_MN2WS_0 /* [ */
/* SPI-0 Data */
static struct spi_mn2ws_data mn2ws_data_0 = {
	.write32            = spi_mn2ws_write32,
	.read32             = spi_mn2ws_read32,
	.base               = SPI_MN2WS_0_BASE,
	.pinsel_cs          = PORT70_SYNC,
	.port_dir           = P7DIR,
	.port_iod           = P7IOD,
	.data_dir           = P7_0_out,
	.pinsel_mode        = P70_mask,
	.cs_high            = P7_0_on,
	.cs_low             = P7_0_off,
	.irq                = SPI_MN2WS_0_IRQ,
};
static struct spi_device mn2ws_ops_0 = {
	.mn2ws_data     = &mn2ws_data_0,
	.modalias       = "MN2WS SPI Adapter(0)",
	.chip_select    = 0,
};
#endif /* ] CONFIG_SPI_MN2WS_0 */

#ifdef CONFIG_SPI_MN2WS_1 /* [ */
static struct spi_mn2ws_data mn2ws_data_1 = {
	.write32            = spi_mn2ws_write32,
	.read32             = spi_mn2ws_read32,
	.base               = SPI_MN2WS_1_BASE,
	.pinsel_cs          = PORT32_SYNC,
	.port_dir           = P3DIR,
	.port_iod           = P3IOD,
	.data_dir           = P3_2_out,
	.pinsel_mode        = P32_mode0,
	.cs_high            = P3_2_on,
	.cs_low             = P3_2_off,
	.irq                = SPI_MN2WS_1_IRQ,
};
static struct spi_device mn2ws_ops_1 = {
	.mn2ws_data     = &mn2ws_data_1,
	.modalias       = "MN2WS SPI Adapter(1)",
	.chip_select    = 1,
};


#endif /* ] CONFIG_SPI_MN2WS_1 */

static struct spi_device *mn2ws_ops[SPI_MN2WS_ADAP_MAX + 1] = {
#ifdef CONFIG_SPI_MN2WS_0 /* [ */
	[0] = &mn2ws_ops_0,
#endif /* ] CONFIG_SPI_MN2WS_0 */
#ifdef CONFIG_SPI_MN2WS_1 /* [ */
	[1] = &mn2ws_ops_1,
#endif /* ] CONFIG_SPI_MN2WS_1 */
	[SPI_MN2WS_ADAP_MAX] = 0,
};

//static int __init mn2ws_spi_probe(struct platform_device *pdev)
static int mn2ws_spi_probe(struct platform_device *pdev)
{
	struct spi_master *master;
	struct mn2ws_spi *mn2ws_spi;
	struct mn2ws_spi_info *mn2ws_spi_info;
	int status = SPI_STATUS_OK;
	
	mn2ws_spi_info = pdev->dev.platform_data;
	printk(KERN_INFO"spi probe mn2ws start\n");
	/* allocate master controler */
	master = spi_alloc_master(&pdev->dev, sizeof(struct mn2ws_spi));
	if(master == NULL){
		printk(KERN_INFO"unable to allocate master\n");
		return -ENOMEM;
	}
	
	if(pdev->id != -1){
		master->bus_num = pdev->id;
	}
	master->mode_bits = 0;

	master->num_chipselect = SPI_N_CHIPSEL;
	master->setup = mn2ws_spi_setup;
	master->transfer = mn2ws_spi_transfer;
	
	dev_set_drvdata(&pdev->dev, master);

	/* get data of master device */
	mn2ws_spi = spi_master_get_devdata(master);
	
	/* mn2ws_spi */
	mn2ws_spi->master = master;
	mn2ws_spi->spi_info = mn2ws_spi_info;
	/*  */
	platform_set_drvdata(pdev, master);
	
	INIT_WORK(&mn2ws_spi->work, mn2ws_spi_work);
	
	spin_lock_init(&mn2ws_spi->lock);
	INIT_LIST_HEAD(&mn2ws_spi->msg_queue);
	
	mn2ws_spi_boardinfo_0.bus_num = master->bus_num; 

	spi_register_board_info(&mn2ws_spi_boardinfo_0, 1);
	
	mn2ws_spi_boardinfo_1.bus_num = master->bus_num;

	spi_register_board_info(&mn2ws_spi_boardinfo_1, 1); 

	/* masterȥϿ */
	status = spi_register_master(master);
	if(status != SPI_STATUS_OK){
		printk(KERN_INFO"spi probe mn2ws register master failed\n");
		return -ENOMEM; // 顼
	}
	printk(KERN_INFO"spi probe mn2ws end \n");
	return 0;
}

static void mn2ws_spi_exit_sub(struct spi_device *spi);
static int mn2ws_spi_suspend( struct platform_device *pdev, pm_message_t state )
{
	int i;

	for (i = 0; i < SPI_MN2WS_ADAP_MAX; ++i){
		if (!mn2ws_ops[i]){
			continue;
		}
		mn2ws_spi_exit_sub(mn2ws_ops[i]);
	}

	return 0;
}

static int  mn2ws_spi_init_sub(struct spi_device *spi);
static int mn2ws_spi_resume( struct platform_device *pdev )
{
	int i, ret =0;

	for (i = 0; i < SPI_MN2WS_ADAP_MAX; ++i){
		if (!mn2ws_ops[i]){
			continue;
		}
		ret = mn2ws_spi_init_sub(mn2ws_ops[i]);
		if (ret < 0){
			break;
		}
	}

	return 0;
}

static struct platform_driver mn2ws_spi_drv = {
	.probe		= mn2ws_spi_probe, /* sLD8 debug */
	.suspend	= mn2ws_spi_suspend,
	.resume		= mn2ws_spi_resume,
	.driver		= {				/* struct device_driver */
		.name   = "spi_mn2ws",
		.owner	= THIS_MODULE,
	},
};

static int  mn2ws_spi_init_sub(struct spi_device *spi)
{
	struct spi_mn2ws_data *mn2ws_device;
	unsigned long base;
	int           irq;
	void *addr;
	unsigned long val;
	unsigned long val_org;
    printk(KERN_INFO"mn2ws spi sub init for dev %d \n",spi->chip_select);
	if (!spi){
		return -ENODEV;
	}
	mn2ws_device = spi->mn2ws_data;
    printk(KERN_INFO"mn2ws spi sub init device \n");
	spi_g_mn2ws_data[spi->chip_select] = spi->mn2ws_data;

	if (!mn2ws_device){
		return -ENODEV;
	}
	base = mn2ws_device->base;
	irq  = mn2ws_device->irq;
    printk(KERN_INFO"mn2ws spi sub irq %d \n",irq);

	init_waitqueue_head(&(mn2ws_device->wait));
	spi_g_mn2ws_data[spi->chip_select]->irq_event = 0;

	spin_lock_init(&(spi_g_mn2ws_data[spi->chip_select]->lock));

        printk(KERN_INFO "spi-mn2ws: i/o base %#08lx. irq %d\n", base, irq);

	if (!request_mem_region(base, IO_SIZE, "spi-mn2ws")){
		printk(KERN_INFO"request memory region error \n");
		goto out;
	}
	mn2ws_device->iomap_base = ioremap(base, IO_SIZE);
	if (!(mn2ws_device->iomap_base)){
		printk(KERN_INFO"iomap base error \n");
		goto out_region;
	}
	mn2ws_device->p_iomap_base = ioremap(PORT_REG_BASE, IO_SIZE);
	if (!(mn2ws_device->p_iomap_base))	{
		printk(KERN_INFO"iomap port base error \n");
		goto out_region;
	}

	mn2ws_device->s_iomap_base = ioremap(PINS_REG_BASE, IO_SIZE);
	if (!(mn2ws_device->s_iomap_base))	{
		printk(KERN_INFO"iomap pin reg base error \n");
		goto out_region;
	}
    printk(KERN_INFO"disable irq start \n");
	disable_irq(irq);
    printk(KERN_INFO"disable irq end \n");
	if (irq > -1){
		if (request_irq(irq, spi_mn2ws_handler, 0, "spi-mn2ws", mn2ws_device) < 0){
			 printk(KERN_INFO"request irq error \n");
			goto out_remap;
		}
	}
	// Clear INTC
	mn_intc_clear( irq );
	 printk(KERN_INFO"clear irq success \n");
	//GxICR(irq) = (GxICR(irq) & 0xFF00) | 0x0001; //commenting for LD4 - to test
#ifdef DEBUG_IO
	DEB3("GxICR(%d) = %x\n", irq, GxICR(irq));
#endif

	disable_irq(irq);
     printk(KERN_INFO"disable irq after clear irq \n");

#ifdef PINSEL
    /* Pinsel register should be set at boot. */
#endif

    /* Port direction set */

#ifdef PORT_CS
	addr = mn2ws_device->p_iomap_base + mn2ws_device->port_dir  ;
	val_org = readb(addr);
	printk(KERN_INFO"val org 0x%lx \n",val_org);
	val  = mn2ws_device->data_dir & val_org;
	printk(KERN_INFO"val is  0x%lx \n",val);
	writeb(val,addr);
#endif
    printk(KERN_INFO"init register \n");
	mn2ws_spi_init_register(spi);
	printk(KERN_INFO"init register end \n");

    /* Verify that the CS is negated */
#ifdef PORT_CS
	spi_cs_negate(spi);
    printk(KERN_INFO"cs negate end \n");
#endif

	return 0;

 out_remap:
	iounmap(mn2ws_device->iomap_base);
 out_region:
	release_mem_region(base, IO_SIZE);
 out:
	return -ENODEV;

}
static int __init mn2ws_spi_init(void )
{
	int i, ret =0;
	int status;
	printk(KERN_INFO"mn2ws init \n");

	for (i = 0; i < SPI_MN2WS_ADAP_MAX; ++i){
		if (!mn2ws_ops[i]){
			continue;
		}
		ret = mn2ws_spi_init_sub(mn2ws_ops[i]);
		if (ret < 0){
			break;
		}
	}
	printk(KERN_INFO"register platform driver modified \n");
	mn2ws_spi_wq = create_singlethread_workqueue(mn2ws_spi_drv.driver.name);
	if (mn2ws_spi_wq == NULL){
		printk(KERN_INFO"create single thread failed \n");
		return -ENOMEM;
	}
	printk(KERN_INFO"register platform driver \n");
#if 0 
	//return platform_driver_probe(&mn2ws_spi_drv, mn2ws_spi_probe);
#else
	status = platform_driver_register(&mn2ws_spi_drv);
	return status;
#endif

}

static void mn2ws_spi_exit_sub(struct spi_device *spi)
{
#ifdef PORT_CS
	spi_cs_negate(spi);
#endif
	free_irq(spi->mn2ws_data->irq, spi->mn2ws_data);
	iounmap(spi->mn2ws_data->iomap_base);
	iounmap(spi->mn2ws_data->p_iomap_base);
	iounmap(spi->mn2ws_data->s_iomap_base);
	release_mem_region(spi->mn2ws_data->base, IO_SIZE);
}

static void __exit mn2ws_spi_exit(void )
{
	int i;

	for (i = 0; i < SPI_MN2WS_ADAP_MAX; ++i){
		if (!mn2ws_ops[i]){
			continue;
		}
		mn2ws_spi_exit_sub(mn2ws_ops[i]);
	}
	flush_workqueue(mn2ws_spi_wq);
	platform_driver_unregister(&mn2ws_spi_drv);
	destroy_workqueue(mn2ws_spi_wq);
}


MODULE_DESCRIPTION("MN2WS SPI driver");
MODULE_LICENSE("GPL");

module_init(mn2ws_spi_init);
module_exit(mn2ws_spi_exit);
